-- CAP-STONE HALL International School - Results Portal Database Schema
-- Run this SQL file to create all necessary tables

CREATE DATABASE IF NOT EXISTS capstone_school CHARACTER SET utf8mb4 COLLATE utf8mb4_unicode_ci;
USE capstone_school;

-- Teachers table (includes admin flag)
CREATE TABLE IF NOT EXISTS teachers (
    id VARCHAR(20) PRIMARY KEY,
    name VARCHAR(100) NOT NULL,
    username VARCHAR(50) UNIQUE NOT NULL,
    password VARCHAR(255) NOT NULL,
    is_admin TINYINT(1) DEFAULT 0,
    classes TEXT,  -- JSON array of class IDs
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;

-- Students table
CREATE TABLE IF NOT EXISTS students (
    id VARCHAR(20) PRIMARY KEY,
    first_name VARCHAR(50) NOT NULL,
    other_names VARCHAR(100),
    class_id VARCHAR(20) NOT NULL,
    sex ENUM('MALE', 'FEMALE') NOT NULL,
    pin VARCHAR(10) NOT NULL,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    INDEX idx_class (class_id),
    INDEX idx_pin (pin)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;

-- Subjects table (per class)
CREATE TABLE IF NOT EXISTS class_subjects (
    id INT AUTO_INCREMENT PRIMARY KEY,
    class_id VARCHAR(20) NOT NULL,
    subject_name VARCHAR(100) NOT NULL,
    display_order INT DEFAULT 0,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    UNIQUE KEY unique_class_subject (class_id, subject_name),
    INDEX idx_class (class_id)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;

-- Student scores (cognitive domain)
CREATE TABLE IF NOT EXISTS student_scores (
    id INT AUTO_INCREMENT PRIMARY KEY,
    student_id VARCHAR(20) NOT NULL,
    subject_name VARCHAR(100) NOT NULL,
    ra_test INT DEFAULT 0,
    mid_term INT DEFAULT 0,
    random_test INT DEFAULT 0,
    practicals INT DEFAULT 0,
    exam INT DEFAULT 0,
    total INT GENERATED ALWAYS AS (ra_test + mid_term + random_test + practicals + exam) STORED,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    FOREIGN KEY (student_id) REFERENCES students(id) ON DELETE CASCADE,
    UNIQUE KEY unique_student_subject (student_id, subject_name)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;

-- Student behaviours (affective domain)
CREATE TABLE IF NOT EXISTS student_behaviours (
    id INT AUTO_INCREMENT PRIMARY KEY,
    student_id VARCHAR(20) NOT NULL,
    behaviour_name VARCHAR(100) NOT NULL,
    rating INT DEFAULT 0 CHECK (rating BETWEEN 0 AND 5),
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    FOREIGN KEY (student_id) REFERENCES students(id) ON DELETE CASCADE,
    UNIQUE KEY unique_student_behaviour (student_id, behaviour_name)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;

-- Student skills (psychomotor domain)
CREATE TABLE IF NOT EXISTS student_skills (
    id INT AUTO_INCREMENT PRIMARY KEY,
    student_id VARCHAR(20) NOT NULL,
    skill_name VARCHAR(100) NOT NULL,
    rating INT DEFAULT 0 CHECK (rating BETWEEN 0 AND 5),
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    FOREIGN KEY (student_id) REFERENCES students(id) ON DELETE CASCADE,
    UNIQUE KEY unique_student_skill (student_id, skill_name)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;

-- Student attendance & comments
CREATE TABLE IF NOT EXISTS student_records (
    id INT AUTO_INCREMENT PRIMARY KEY,
    student_id VARCHAR(20) UNIQUE NOT NULL,
    days_opened INT DEFAULT 0,
    days_present INT DEFAULT 0,
    days_absent INT GENERATED ALWAYS AS (days_opened - days_present) STORED,
    resumption_date VARCHAR(100),
    instructor_comment TEXT,
    principal_comment TEXT,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    FOREIGN KEY (student_id) REFERENCES students(id) ON DELETE CASCADE
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;

-- School settings (academic year, term, etc.)
CREATE TABLE IF NOT EXISTS school_settings (
    id INT AUTO_INCREMENT PRIMARY KEY,
    setting_key VARCHAR(50) UNIQUE NOT NULL,
    setting_value TEXT,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;

-- Insert default school settings
INSERT INTO school_settings (setting_key, setting_value) VALUES
('academic_year', '2024/2025'),
('current_term', '3RD TERM')
ON DUPLICATE KEY UPDATE setting_value=VALUES(setting_value);

-- Insert default teacher (admin)
INSERT INTO teachers (id, name, username, password, is_admin, classes) VALUES
('t001', 'Form Teacher', 'teacher', '$2y$10$92IXUNpkjO0rOQ5byMi.Ye4oKoEa3Ro9llC/.og/at2.uheWG/igi', 1, '["sss1","sss2"]')
ON DUPLICATE KEY UPDATE name=VALUES(name);
-- Default password: school123 (hashed with bcrypt)

-- Insert default students for 10th Grade (SSS1)
INSERT INTO students (id, first_name, other_names, class_id, sex, pin) VALUES
('s001', 'Christopher', 'Oghenekome Divine', 'sss1', 'MALE', '1001'),
('s002', 'O.', 'Tejiri', 'sss1', 'FEMALE', '1002'),
('s003', 'O.', 'Success', 'sss1', 'FEMALE', '1003'),
('s004', 'O.', 'Rhema', 'sss1', 'FEMALE', '1004'),
('s005', 'O.', 'Chidera', 'sss1', 'FEMALE', '1005'),
('s006', 'U.', 'Prevail', 'sss1', 'MALE', '1006'),
('s007', 'A.', 'Diadem', 'sss1', 'MALE', '1007'),
('s008', 'O.', 'Christabel', 'sss1', 'FEMALE', '1008'),
('s009', 'E.', 'Blessing', 'sss1', 'FEMALE', '1009'),
('s010', 'M.', 'Justina', 'sss1', 'FEMALE', '1010'),
('s011', 'E.', 'Golden', 'sss1', 'MALE', '1011'),
('s012', 'U.', 'Prudent', 'sss1', 'MALE', '1012'),
('s013', 'O.', 'Funmilayo', 'sss1', 'FEMALE', '1013'),
('s014', 'J.', 'Peculiar', 'sss1', 'FEMALE', '1014'),
('s015', 'Napoleon', 'Grace', 'sss2', 'FEMALE', '2001')
ON DUPLICATE KEY UPDATE first_name=VALUES(first_name);

-- Insert default subjects for senior classes (10th-12th Grade)
INSERT INTO class_subjects (class_id, subject_name, display_order) VALUES
('sss1', 'English Studies', 1),
('sss1', 'Mathematics', 2),
('sss1', 'Civic Education', 3),
('sss1', 'Economics', 4),
('sss1', 'Geography', 5),
('sss1', 'Government', 6),
('sss1', 'Literature', 7),
('sss1', 'Marketing', 8),
('sss1', 'Agricultural Science', 9),
('sss1', 'Biology', 10),
('sss1', 'Chemistry', 11),
('sss1', 'Data Processing', 12),
('sss1', 'Further Mathematics', 13),
('sss1', 'Physics', 14),
('sss1', 'Commerce', 15),
('sss1', 'Financial Accounting', 16),
('sss1', 'C.R.S', 17)
ON DUPLICATE KEY UPDATE subject_name=VALUES(subject_name);

-- Sample scores for Christopher (s001)
INSERT INTO student_scores (student_id, subject_name, ra_test, mid_term, random_test, practicals, exam) VALUES
('s001', 'English Studies', 5, 4, 9, 7, 33),
('s001', 'Mathematics', 8, 10, 9, 7, 48),
('s001', 'Civic Education', 7, 8, 9, 9, 46),
('s001', 'Economics', 10, 5, 6, 4, 30),
('s001', 'Geography', 8, 6, 7, 7, 39),
('s001', 'Government', 4, 6, 9, 9, 49),
('s001', 'Literature', 7, 8, 10, 7, 35),
('s001', 'Marketing', 6, 7, 6, 7, 40),
('s001', 'Agricultural Science', 7, 9, 10, 10, 48),
('s001', 'Biology', 10, 7, 10, 10, 41),
('s001', 'Chemistry', 8, 9, 10, 10, 45),
('s001', 'Data Processing', 7, 8, 9, 8, 36),
('s001', 'Further Mathematics', 6, 9, 9, 9, 50),
('s001', 'Physics', 9, 8, 8, 8, 26),
('s001', 'Commerce', 7, 7, 10, 7, 44),
('s001', 'Financial Accounting', 8, 8, 7, 4, 39),
('s001', 'C.R.S', 10, 9, 5, 5, 53)
ON DUPLICATE KEY UPDATE ra_test=VALUES(ra_test), mid_term=VALUES(mid_term), random_test=VALUES(random_test), practicals=VALUES(practicals), exam=VALUES(exam);

-- Sample behaviours for Christopher
INSERT INTO student_behaviours (student_id, behaviour_name, rating) VALUES
('s001', 'Attendance in Class', 5),
('s001', 'Attentiveness', 4),
('s001', 'Honesty', 5),
('s001', 'Initiative', 4),
('s001', 'Neatness', 4),
('s001', 'Participation in School Activities', 5),
('s001', 'Politeness', 4),
('s001', 'Relationship With Others', 5),
('s001', 'Sense of Responsibility', 5)
ON DUPLICATE KEY UPDATE rating=VALUES(rating);

-- Sample skills for Christopher
INSERT INTO student_skills (student_id, skill_name, rating) VALUES
('s001', 'Art & Craft', 5),
('s001', 'Fluency', 4),
('s001', 'Hand Writing', 4),
('s001', 'Indoor/Outdoor Games', 5),
('s001', 'Tools Handling', 4)
ON DUPLICATE KEY UPDATE rating=VALUES(rating);

-- Sample attendance for Christopher
INSERT INTO student_records (student_id, days_opened, days_present, resumption_date, instructor_comment, principal_comment) VALUES
('s001', 128, 128, 'Monday 5th January, 2026', 'He needs to develop self control and handle disagreement respectfully.', 'Satisfactory.')
ON DUPLICATE KEY UPDATE days_opened=VALUES(days_opened), days_present=VALUES(days_present);
